<?php

namespace ElementorFloatingMenu\Widgets;

use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Group_Control_Border;
use Elementor\Icons_Manager;
use Elementor\Plugin;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Exception;

if (!defined('ABSPATH')) {
    exit;
} // Exit if accessed directly

/**
 * Elementor Floating_Menu
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Floating_Menu extends Widget_Base
{
    protected static $post_types = array();

    /**
     * Retrieve the widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     *
     * @access public
     *
     */
    public function get_name()
    {
        return 'floating-menu';
    }

    /**
     * Display Conditions
     *
     * Holds all the conditions for display on the frontend
     *
     * @since 1.2.7
     * @access protected
     *
     * @var bool
     */
    protected $conditions = [];

    /**
     * Retrieve the widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     *
     * @access public
     *
     */
    public function get_title()
    {
        return __('Floating Menu', 'elementor-floating-menu');
    }

    /**
     * Retrieve the widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     *
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-menu-bar';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @return array Widget categories.
     * @since 1.0.0
     *
     * @access public
     *
     */
    public function get_categories()
    {
        return ['general'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @return array Widget scripts dependencies.
     * @since 1.0.0
     *
     * @access public
     *
     */
    public function get_script_depends()
    {
        return ['elementor-floating-menu'];
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords() {
        return [ 'floating', 'icon', 'link' ];
    }

    /**
     * Floating_Menu constructor.
     *
     * @param array $data
     * @param null $args
     * @throws Exception
     */
    public function __construct($data = [], $args = null)
    {
        parent::__construct($data, $args);
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function _register_controls()
    {
        $this->_register_control_header();

        $this->_register_control_list_item();

        $this->_register_control_condition();

        $this->_register_control_style();
    }

    /**
     * Register the widget control Header.
     *
     * @since 1.0.0
     *
     * @access private
     */
    private function _register_control_header() : void
    {
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Floating Menu', 'elementor-floating-menu'),
            ]
        );
    
        $this->add_control(
            'title_open',
            [
                'label' => __('Caption when menu is open', 'elementor-floating-menu'),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Contact', 'elementor-floating-menu' ),
            ]
        );

        $this->add_control(
            'title_close',
            [
                'label' => __('Caption when menu is close', 'elementor-floating-menu'),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Close', 'elementor-floating-menu' ),
            ]
        );

        $this->add_control(
            'icon',
            [
                'label' => __('Choose Icon', 'elementor-floating-menu'),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fab fa-wpbeginner',
                    'library' => 'solid',
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Register the widget control Condition show widget.
     *
     * @since 1.0.0
     *
     * @access private
     */
    private function _register_control_condition() : void
    {
        global $wp_roles;

        $default_date_start = date( 'Y-m-d', strtotime( '-3 day' ) + ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
        $default_date_end 	= date( 'Y-m-d', strtotime( '+3 day' ) + ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
        $default_interval 	= $default_date_start . ' to ' . $default_date_end;

        $this->start_controls_section(
            'section_content_display_conditions',
            [
                'label' => __('Display Conditions', 'elementor-floating-menu')
            ]
        );

        $this->add_control(
            'display_conditions_enable',
            [
                'label'						=> __( 'Display Conditions', 'elementor-floating-menu' ),
                'type' 						=> Controls_Manager::SWITCHER,
                'default' 					=> '',
                'label_on' 					=> __( 'Yes', 'elementor-floating-menu' ),
                'label_off' 				=> __( 'No', 'elementor-floating-menu' ),
                'return_value'				=> 'yes',
                'frontend_available'		=> true,
            ]
        );

        $this->add_control(
            'display_conditions_relation',
            [
                'label'						=> __( 'Display on', 'elementor-floating-menu' ),
                'type'						=> Controls_Manager::SELECT,
                'default'					=> 'all',
                'options'					=> [
                    'all' 		=> __( 'All conditions met', 'elementor-floating-menu' ),
                    'any' 		=> __( 'Any condition met', 'elementor-floating-menu' ),
                ],
                'condition'					=> [
                    'display_conditions_enable' => 'yes',
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'condition_key',
            [
                'type' 			=> Controls_Manager::SELECT,
                'default' 		=> 'authentication',
                'label_block'	=> true,
                'groups' 		=> [
                    [
                        'label'		=> __( 'User', 'elementor-floating-menu' ),
                        'options' 	=> [
                            'authentication' 	=> __( 'Login Status', 'elementor-floating-menu' ),
                            'role' 				=> __( 'User Role', 'elementor-floating-menu' ),
                        ],
                    ],
                    [
                        'label'					=> __( 'Singular', 'elementor-floating-menu' ),
                        'options' 				=> [
                            'page' 				=> __( 'Page', 'elementor-floating-menu' ),
                            'post' 				=> __( 'Post', 'elementor-floating-menu' ),
                            'static_page' 		=> __( 'Special (404, Home, Front, Blog)', 'elementor-floating-menu' ),
                            'post_type' 		=> __( 'Post Type', 'elementor-floating-menu' ),
                        ],
                    ],
                    [
                        'label'					=> __( 'Archive', 'elementor-floating-menu' ),
                        'options' 				=> [
                            'date_archive'		=> __( 'Date', 'elementor-floating-menu' ),
                            'author_archive'	=> __( 'Author', 'elementor-floating-menu' ),
                        ],
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'condition_operator',
            [
                'type' 			=> Controls_Manager::SELECT,
                'default' 		=> 'is',
                'label_block' 	=> true,
                'options' 		=> [
                    'is' 		=> __( 'Is', 'elementor-floating-menu' ),
                    'not' 		=> __( 'Is not', 'elementor-floating-menu' ),
                ],
            ]
        );

        $repeater->add_control(
            'condition_authentication_value',
            [
                'type' 		=> Controls_Manager::SELECT,
                'default' 	=> 'authenticated',
                'label_block' => true,
                'options' 	=> [
                    'authenticated' => __( 'Logged in', 'elementor-floating-menu' ),
                ],
                'condition' => [
                    'condition_key' => 'authentication',
                ],
            ]
        );;

        $repeater->add_control(
            'condition_role_value',
            [
                'type' 			=> Controls_Manager::SELECT,
                'description' 	=> __( 'Warning: This condition applies only to logged in visitors.', 'elementor-floating-menu' ),
                'default' 		=> 'subscriber',
                'label_block' 	=> true,
                'options' 		=> $wp_roles->get_names(),
                'condition' 	=> [
                    'condition_key' => 'role',
                ],
            ]
        );

        $repeater->add_control(
            'condition_date_value',
            [
                'label'				=> __( 'In interval', 'elementor-floating-menu' ),
                'type'				=> Controls_Manager::DATE_TIME,
                'picker_options'	=> [
                    'enableTime'	=> false,
                    'mode' 			=> 'range',
                ],
                'label_block'		=> true,
                'default'			=> $default_interval,
                'condition'			=> [
                    'condition_key' => 'date',
                ],
            ]
        );

        $repeater->add_control(
            'condition_time_value',
            [
                'label'				=> __( 'Before', 'elementor-floating-menu' ),
                'type'				=> Controls_Manager::DATE_TIME,
                'picker_options'	=> [
                    'dateFormat' 	=> "H:i",
                    'enableTime' 	=> true,
                    'noCalendar' 	=> true,
                ],
                'label_block'		=> true,
                'default' 			=> '',
                'condition' 		=> [
                    'condition_key' => 'time',
                ],
            ]
        );

        $pages_all = $this->get_all_posts_by_type( 'page' );
        $posts_all = $this->get_all_posts_by_type( 'post' );

        $repeater->add_control(
            'condition_page_value',
            [
                'type'				=> Controls_Manager::SELECT2,
                'default' 		=> '',
                'placeholder'	=> __( 'Any', 'elementor-floating-menu' ),
                'description'	=> __( 'Leave blank for any page.', 'elementor-floating-menu' ),
                'label_block' 	=> true,
                'multiple'		=> true,
                'options'			=> $pages_all,
                'condition' 	=> [
                    'condition_key' => 'page',
                ],
            ]
        );

        $repeater->add_control(
            'condition_post_value',
            [
                'type'				=> Controls_Manager::SELECT2,
                'default' 		=> '',
                'placeholder'	=> __( 'Any', 'elementor-floating-menu' ),
                'description'	=> __( 'Leave blank for any post.', 'elementor-floating-menu' ),
                'label_block' 	=> true,
                'multiple'		=> true,
                'options'			=> $posts_all,
                'condition' 	=> [
                    'condition_key' => 'post',
                ],
            ]
        );

        $repeater->add_control(
            'condition_static_page_value',
            [
                'type' 			=> Controls_Manager::SELECT,
                'default' 		=> 'home',
                'label_block' 	=> true,
                'options' 		=> [
                    'home'		=> __( 'Homepage', 'elementor-floating-menu' ),
                    'static'	=> __( 'Front Page', 'elementor-floating-menu' ),
                    'blog'		=> __( 'Blog', 'elementor-floating-menu' ),
                    '404'		=> __( '404 Page', 'elementor-floating-menu' ),
                ],
                'condition' 	=> [
                    'condition_key' => 'static_page',
                ],
            ]
        );

        $repeater->add_control(
            'condition_post_type_value',
            [
                'type' 			=> Controls_Manager::SELECT2,
                'default' 		=> '',
                'placeholder'	=> __( 'Any', 'elementor-floating-menu' ),
                'description'	=> __( 'Leave blank or select all for any post type.', 'elementor-floating-menu' ),
                'label_block' 	=> true,
                'multiple'		=> true,
                'options' 		=> $this->get_post_types(),
                'condition' 	=> [
                    'condition_key' => 'post_type',
                ],
            ]
        );

        $repeater->add_control(
            'condition_date_archive_value',
            [
                'type' 			=> Controls_Manager::SELECT2,
                'default' 		=> '',
                'placeholder'	=> __( 'Any', 'elementor-floating-menu' ),
                'description'	=> __( 'Leave blank or select all for any date based archive.', 'elementor-floating-menu' ),
                'multiple'		=> true,
                'label_block' 	=> true,
                'options' 		=> [
                    'day'		=> __( 'Day', 'elementor-floating-menu' ),
                    'month'		=> __( 'Month', 'elementor-floating-menu' ),
                    'year'		=> __( 'Year', 'elementor-floating-menu' ),
                ],
                'condition' 	=> [
                    'condition_key' => 'date_archive',
                ],
            ]
        );

        $repeater->add_control(
            'condition_author_archive_value',
            [
                'type'				=> Controls_Manager::SELECT2,
                'default' 		=> '',
                'placeholder'	=> __( 'Any', 'elementor-floating-menu' ),
                'description'	=> __( 'Leave blank for all authors.', 'elementor-floating-menu' ),
                'multiple'		=> true,
                'label_block' 	=> true,
                'options'			=> $this->get_users(),
                'condition' 	=> [
                    'condition_key' => 'author_archive',
                ],
            ]
        );

        $this->add_control(
            'display_conditions',
            [
                'label' 	=> __( 'Conditions', 'elementor-floating-menu' ),
                'type' 		=> Controls_Manager::REPEATER,
                'default' 	=> [
                    [
                        'condition_key' 					=> 'authentication',
                        'condition_operator' 			=> 'is',
                        'condition_authentication_value' => 'authenticated',
                    ],
                ],
                'condition'		=> [
                    'display_conditions_enable' => 'yes',
                ],
                'fields' 		=> array_values( $repeater->get_controls() ),
                'title_field' 	=> 'Condition',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Register the widget control list Item.
     *
     * @since 1.0.0
     *
     * @access private
     */
    private function _register_control_list_item() : void
    {
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Floating Menu', 'elementor-floating-menu'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'list_title',
            [
                'label' => __('Caption', 'elementor-floating-menu'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Enter text', 'elementor-floating-menu'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'floating_icon',
            [
                'label' => __( 'Icon', 'elementor-floating-menu' ),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'floating',
                'default' => [
                    'value' => 'fab fa-wordpress',
                    'library' => 'fa-brands',
                ],
                'recommended' => [
                    'fa-brands' => [
                        'android',
                        'apple',
                        'behance',
                        'bitbucket',
                        'codepen',
                        'delicious',
                        'deviantart',
                        'digg',
                        'dribbble',
                        'elementor',
                        'facebook',
                        'flickr',
                        'foursquare',
                        'free-code-camp',
                        'github',
                        'gitlab',
                        'globe',
                        'houzz',
                        'instagram',
                        'jsfiddle',
                        'linkedin',
                        'medium',
                        'meetup',
                        'mix',
                        'mixcloud',
                        'odnoklassniki',
                        'pinterest',
                        'product-hunt',
                        'reddit',
                        'shopping-cart',
                        'skype',
                        'slideshare',
                        'snapchat',
                        'soundcloud',
                        'spotify',
                        'stack-overflow',
                        'steam',
                        'telegram',
                        'thumb-tack',
                        'tripadvisor',
                        'tumblr',
                        'twitch',
                        'twitter',
                        'viber',
                        'vimeo',
                        'vk',
                        'weibo',
                        'weixin',
                        'whatsapp',
                        'wordpress',
                        'xing',
                        'yelp',
                        'youtube',
                        '500px',
                    ],
                    'fa-solid' => [
                        'envelope',
                        'link',
                        'rss',
                    ],
                ],
            ]
        );

        $repeater->add_control(
            'link',
            [
                'label' => __( 'Link', 'elementor-floating-menu' ),
                'type' => Controls_Manager::URL,
                'default' => [
                    'is_external' => 'true',
                ],
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => __( 'https://your-link.com', 'elementor-floating-menu' ),
            ]
        );

        $repeater->add_control(
            'item_icon_color',
            [
                'label' => __( 'Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'default',
                'options' => [
                    'default' => __( 'Official Color', 'elementor-floating-menu' ),
                    'custom' => __( 'Custom', 'elementor-floating-menu' ),
                ],
            ]
        );

        $repeater->add_control(
            'item_icon_primary_color',
            [
                'label' => __( 'Primary Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'item_icon_color' => 'custom',
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}.elementor-social-icon' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $repeater->add_control(
            'item_icon_secondary_color',
            [
                'label' => __( 'Secondary Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'item_icon_color' => 'custom',
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}.elementor-social-icon i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} {{CURRENT_ITEM}}.elementor-social-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'floating_icon_list',
            [
                'label' => __( 'Floating Icons', 'elementor-floating-menu' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'list_title' => __('Enter text', 'elementor-floating-menu'),
                    ],
                ],
                'title_field' => '{{{ list_title }}}',
            ]
        );

        $this->add_control(
            'shape',
            [
                'label' => __( 'Shape', 'elementor-floating-menu' ),
                'type' => Controls_Manager::HIDDEN,
                'default' => 'circle',
                'options' => [
                    'circle' => __( 'Circle', 'elementor-floating-menu' ),
                ],
                'prefix_class' => 'elementor-shape-',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Register the widget control Style icon.
     *
     * @since 1.0.0
     *
     * @access private
     */
    private function _register_control_style(): void
    {
        $this->start_controls_section(
            'section_floating_style',
            [
                'label' => __('Style Section', 'elementor-floating-menu'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => __('Caption Text Color', 'elementor-floating-menu'),
                'type' => Controls_Manager::COLOR,
                'default' => '#000',
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => __('Caption Background Color', 'elementor-floating-menu'),
                'type' => Controls_Manager::COLOR,
                'default' => '#FFF',
            ]
        );

        $this->add_control(
            'font_size_text',
            [
                'label' => __('Size of text', 'elementor-floating-menu'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 25,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 18,
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'selected_icon',
            [
                'label' => __('Icon', 'elementor-floating-menu'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __( 'Icon Color Action', 'elementor-floating-menu' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'default',
                'options' => [
                    'default' => __( 'Official Color', 'elementor-floating-menu' ),
                    'custom' => __( 'Custom', 'elementor-floating-menu' ),
                ],
            ]
        );

        $this->add_control(
            'icon_primary_color',
            [
                'label' => __( 'Primary Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'icon_color' => 'custom',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-grid-item-trigger' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_secondary_color',
            [
                'label' => __( 'Secondary Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'icon_color' => 'custom',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-grid-item-action i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-grid-item-action svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __( 'Size', 'elementor-floating-menu' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 6,
                        'max' => 300,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}}' => '--icon-size: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_padding',
            [
                'label' => __( 'Padding', 'elementor-floating-menu' ),
                'type' => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .elementor-social-icon' => '--icon-padding: {{SIZE}}{{UNIT}}',
                ],
                'default' => [
                    'unit' => 'em',
                ],
                'tablet_default' => [
                    'unit' => 'em',
                ],
                'mobile_default' => [
                    'unit' => 'em',
                ],
                'range' => [
                    'em' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => __( 'Spacing', 'elementor-floating-menu' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}}' => '--grid-column-gap: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'row_gap',
            [
                'label' => __( 'Rows Gap', 'elementor-floating-menu' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}}' => '--grid-row-gap: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'image_border', // We know this mistake - TODO: 'icon_border' (for hover control condition also)
                'selector' => '{{WRAPPER}} .elementor-social-icon',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'border_radius',
            [
                'label' => __( 'Border Radius', 'elementor-floating-menu' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_social_hover',
            [
                'label' => __( 'Icon Hover', 'elementor-floating-menu' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'hover_primary_color',
            [
                'label' => __( 'Primary Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'condition' => [
                    'icon_color' => 'custom',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-social-icon:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'hover_secondary_color',
            [
                'label' => __( 'Secondary Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'condition' => [
                    'icon_color' => 'custom',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-social-icon:hover i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-social-icon:hover svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'hover_border_color',
            [
                'label' => __( 'Border Color', 'elementor-floating-menu' ),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'condition' => [
                    'image_border_border!' => '',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-social-icon:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'hover_animation',
            [
                'label' => __( 'Hover Animation', 'elementor-floating-menu' ),
                'type' => Controls_Manager::HOVER_ANIMATION,
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Get All Posts by Post Type.
     *
     * @param string $post_type Post type.
     * @return array
     * @access public
     */
    public function get_all_posts_by_type(string $post_type): array
    {

        $post_list = get_posts(
            array(
                'post_type'      => $post_type,
                'orderby'        => 'date',
                'order'          => 'DESC',
                'posts_per_page' => -1,
            )
        );

        $posts = array();

        if ( ! empty( $post_list ) && ! is_wp_error( $post_list ) ) {
            foreach ( $post_list as $post ) {
                $posts[ $post->ID ] = $post->post_title;
            }
        }

        return $posts;
    }

    /**
     * Get list of users.
     *
     * @uses   get_users()
     * @link   https://codex.wordpress.org/Function_Reference/get_users
     * @return array $user_list data for all users.
     */
    public static function get_users() {

        $users     = get_users();
        $user_list = array();

        if ( empty( $users ) ) {
            return $user_list;
        }

        foreach ( $users as $user ) {
            $user_list[ $user->ID ] = $user->display_name;
        }

        return $user_list;
    }

    /**
     * Get Post Types.
     *
     * @access public
     */
    public static function get_post_types(): array
    {
        if ( ! empty( self::$post_types ) ) {
            return self::$post_types;
        }

        $post_types = get_post_types(
            array(
                'public' => true,
            ),
            'objects'
        );

        $options = array();

        foreach ( $post_types as $post_type ) {
            $options[ $post_type->name ] = $post_type->label;
        }

        self::$post_types = $options;

        return $options;
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        if ( 'yes' === $settings[ 'display_conditions_enable' ] ) {

            // Set the conditions
            $this->set_conditions( $this->get_id(), $settings['display_conditions'] );

            if ( $this->is_visible( $this->get_id(), $settings['display_conditions_relation'] ) ) { // Check the conditions
                $this->render_content_menu($settings);
            }
        } else {
            //render content when display conditions is disable
            $this->render_content_menu($settings);
        }
    }
    
    
    /**
     * Render content floating menu
     *
     * @param $settings
     */
    private function render_content_menu($settings) {
        $fallback_defaults = [
            'fa fa-facebook',
            'fa fa-twitter',
            'fa fa-google-plus',
        ];
    
        $class_animation = '';
    
        if ( ! empty( $settings['hover_animation'] ) ) {
            $class_animation = ' elementor-animation-' . $settings['hover_animation'];
        }
    
        $font_size = $settings['font_size_text']['size'] . $settings['font_size_text']['unit'];
        $migration_allowed = Icons_Manager::is_migration_allowed();
        ?>
        <div class="flm">
            <div class="elementor-social-icons-wrapper elementor-grid">
                <?php
                foreach ( $settings['floating_icon_list'] as $index => $item ) {
                    $migrated = isset( $item['__fa4_migrated']['floating_icon'] );
                    $is_new = empty( $item['floating'] ) && $migration_allowed;
                    $floating = '';
                
                    // add old default
                    if ( empty( $item['floating'] ) && ! $migration_allowed ) {
                        $item['floating'] = isset( $fallback_defaults[ $index ] ) ? $fallback_defaults[ $index ] : 'fa fa-wordpress';
                    }
                
                    if ( ! empty( $item['floating'] ) ) {
                        $floating = str_replace( 'fa fa-', '', $item['floating'] );
                    }
                
                    if ( ( $is_new || $migrated ) && 'svg' !== $item['floating_icon']['library'] ) {
                        $floating = explode( ' ', $item['floating_icon']['value'], 2 );
                        if ( empty( $floating[1] ) ) {
                            $floating = '';
                        } else {
                            $floating = str_replace( 'fa-', '', $floating[1] );
                        }
                    }
                    if ( 'svg' === $item['floating_icon']['library'] ) {
                        $floating = get_post_meta( $item['floating_icon']['value']['id'], '_wp_attachment_image_alt', true );
                    }
                
                    $link_key = 'link_' . $index;
                
                    $this->add_render_attribute( $link_key, 'class', [
                        'elementor-icon',
                        'elementor-social-icon',
                        'elementor-social-icon-' . $floating . $class_animation,
                        'elementor-repeater-item-' . $item['_id'],
                    ] );
                
                    $this->add_link_attributes( $link_key, $item['link'] );
                
                    ?>
                    <div class="elementor-grid-item floating-menu-item" style="display: none">
                        <a <?php echo $this->get_render_attribute_string( $link_key ); ?>>
                            <span class="elementor-screen-only"><?php echo ucwords( $floating ); ?></span>
                            <?php
                            if ( $is_new || $migrated ) {
                                Icons_Manager::render_icon( $item['floating_icon'] );
                            } else { ?>
                                <i class="<?php echo esc_attr( $item['floating'] ); ?>"></i>
                            <?php } ?>
                        </a>
                        <?php if($item['list_title']) { ?>
                            <div class="floating-menu-title" style="font-size: <?php echo $font_size; ?>">
                                <p><?php echo $item['list_title'] ?></p>
                            </div>
                        <?php } ?>
                    </div>
                <?php } ?>
                <div class="elementor-grid-item elementor-grid-item-action">
                    <div class="floating-menu-trigger-open" style="display: block">
                        <a class="elementor-icon elementor-social-icon elementor-grid-item-trigger">
                            <span class="elementor-screen-only"></span>
                            <?php
                            Icons_Manager::render_icon( $settings['icon'] );
                            ?>
                        </a>
                        <?php if($settings['title_open']) { ?>
                            <div class="floating-menu-title actions" style="background-color: <?php echo $settings['background_color'];?> !important; font-size: <?php echo $font_size;?> !important;">
                                <p style="color: <?php echo $settings['text_color'];?> !important;"><?php echo $settings['title_open'] ?></p>
                            </div>
                        <?php } ?>
                    </div>
                    <div class="floating-menu-trigger-close" style="display: none">
                        <a class="elementor-icon elementor-social-icon elementor-grid-item-trigger">
                            <span class="elementor-screen-only"></span>
                            <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 1000 1000" xml:space="preserve">
                                <metadata> Svg Vector Icons : http://www.onlinewebfonts.com/icon </metadata>
                                <g>
                                    <path d="M629.3,499.1l333.5-331.8c18.1-18.1,27.2-40,27.2-65.5c0-25.5-9.1-47.1-27.2-64.7c-18.1-17.6-40-26.4-65.5-26.4s-47.4,8.5-65.5,25.5L500,369.8L168.2,36.4c-9.1-7.9-19.3-14.2-30.6-18.7s-23-6.8-34.9-6.8s-23.5,2.3-34.9,6.8S46.3,28.4,37.2,36.4C19.1,54.5,10,76.4,10,101.9c0,25.5,9.1,47.4,27.2,65.5l333.5,331.8L37.2,832.6c-18.1,17-27.2,38.3-27.2,63.8c0,25.5,9.1,47.4,27.2,65.5c18.1,18.1,40,27.2,65.5,27.2c25.5,0,47.4-9.1,65.5-27.2L500,630.2l331.8,331.8c18.1,18.1,40,27.2,65.5,27.2s47.4-9.1,65.5-27.2c18.1-18.1,27.2-40,27.2-65.5c0-25.5-9.1-46.8-27.2-63.8L629.3,499.1z">
                                    </path>
                                </g>
                            </svg>
                        </a>
                        <?php if($settings['title_close']) { ?>
                            <div class="floating-menu-title" style="background-color: <?php echo $settings['background_color'];?> !important; font-size: <?php echo $font_size;?> !important;">
                                <p style="color: <?php echo $settings['text_color'];?> !important;"><?php echo $settings['title_close'] ?></p>
                            </div>
                        <?php } ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Check conditions.
     *
     * Checks for all or any conditions and returns true or false
     * depending on weather the content can be shown or not
     *
     * @access protected
     * @static
     *
     * @param $id
     * @param mixed $relation Required conditions relation
     *
     * @return bool
     */
    protected function is_visible( $id, $relation ): bool
    {
        if ( ! Plugin::$instance->editor->is_edit_mode() ) {
            if ( 'any' === $relation ) {
                if ( ! in_array( true, $this->conditions[ $id ] ) )
                    return false;
            } else {
                if ( in_array( false, $this->conditions[ $id ] ) )
                    return false;
            }
        }

        return true;
    }

    /**
     * Set conditions.
     *
     * Sets the conditions property to all conditions comparison values
     *
     * @param $id
     * @param mixed $conditions The conditions from the repeater field control
     *
     * @return void
     * @access protected
     * @static
     *
     */
    protected function set_conditions( $id, $conditions = [] ) {
        if ( ! $conditions )
            return;

        foreach ( $conditions as $index => $condition ) {
            $key 		= $condition['condition_key'];
            $operator 	= $condition['condition_operator'];
            $value 		= $condition['condition_' . $key . '_value'];

            if ( method_exists( $this, 'check_' . $key ) ) {
                $check = call_user_func( [ $this, 'check_' . $key ], $value, $operator );
                $this->conditions[ $id ][ $key . '_' . $condition['_id'] ] = $check;
            }
        }
    }

    /**
     * Compare conditions.
     *
     * Checks two values against an operator
     *
     * @access protected
     * @static
     *
     * @param mixed $left_value First value to compare.
     * @param mixed $right_value Second value to compare.
     * @param string $operator Comparison operator.
     *
     * @return bool
     */
    protected static function compare($left_value, $right_value, string $operator) {
        switch ( $operator ) {
            case 'is':
                return $left_value == $right_value;
            case 'not':
                return $left_value != $right_value;
            default:
                return $left_value === $right_value;
        }
    }

    /**
     * Check user login status
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     *
     * @access protected
     */
    protected static function check_authentication($value, string $operator) {
        return self::compare( is_user_logged_in(), true, $operator );
    }

    /**
     * Check user role
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     *
     * @access protected
     */
    protected static function check_role($value, string $operator) {

        $user = wp_get_current_user();
        return self::compare( is_user_logged_in() && in_array( $value, $user->roles ), true, $operator );
    }

    /**
     * Check current page
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     * @since 2.1.0
     *
     * @access protected
     */
    protected static function check_page($value, string $operator) {
        $show = false;

        if ( is_array( $value ) && ! empty( $value ) ) {
            foreach ( $value as $_key => $_value ) {
                if ( is_page( $_value ) ) {
                    $show = true; break;
                }
            }
        } else { $show = is_page( $value ); }

        return self::compare( $show, true, $operator );
    }

    /**
     * Check current post
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     * @since 1.2.7
     *
     * @access protected
     */
    protected static function check_post($value, string $operator) {
        $show = false;

        if ( is_array( $value ) && ! empty( $value ) ) {
            foreach ( $value as $_key => $_value ) {
                if ( is_single( $_value ) || is_singular( $_value ) ) {
                    $show = true; break;
                }
            }
        } else { $show = is_single( $value ) || is_singular( $value ); }

        return self::compare( $show, true, $operator );
    }

    /**
     * Check static page
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     * @since 1.2.7
     *
     * @access protected
     */
    protected static function check_static_page($value, string $operator)
    {
        if ( 'home' === $value ) {
            return self::compare( ( is_front_page() && is_home() ), true, $operator );
        } elseif ( 'static' === $value ) {
            return self::compare( ( is_front_page() && ! is_home() ), true, $operator );
        } elseif ( 'blog' === $value ) {
            return self::compare( ( ! is_front_page() && is_home() ), true, $operator );
        } elseif ( '404' === $value ) {
            return self::compare( is_404(), true, $operator );
        }
    }

    /**
     * Check current post type
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     * @since 1.2.7
     *
     * @access protected
     */
    protected static function check_post_type($value, string $operator) {
        $show = false;

        if ( is_array( $value ) && ! empty( $value ) ) {
            foreach ( $value as $_key => $_value ) {
                if ( is_singular( $_value ) ) {
                    $show = true; break;
                }
            }
        } else { $show = is_singular( $value ); }

        return self::compare( $show, true, $operator );
    }

    /**
     * Check current date archive
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     *
     * @access protected
     */
    protected static function check_date_archive($value, string $operator) {
        $show = false;

        if ( is_array( $value ) && ! empty( $value ) ) {
            foreach ( $value as $_key => $_value ) {
                if ( self::check_date_archive_type( $_value ) ) {
                    $show = true; break;
                }
            }
        } else { $show = is_date( $value ); }

        return self::compare( $show, true, $operator );
    }

    /**
     * Checks a given date type against the current page template
     *
     * @param string $type The type of date archive to check against
     * @return bool
     *
     * @access protected
     */
    protected static function check_date_archive_type(string $type) {
        if ( 'day' === $type ) { // Day
            return is_day();
        } elseif ( 'month' === $type ) { // Month
            return is_month();
        } elseif ( 'year' === $type ) { // Year
            return is_year();
        }

        return false;
    }

    /**
     * Check current author archive
     *
     * @param mixed $value The control value to check
     * @param string $operator Comparison operator.
     * @return bool
     *
     * @access protected
     */
    protected static function check_author_archive($value, string $operator) {
        $show = false;

        if ( is_array( $value ) && ! empty( $value ) ) {
            foreach ( $value as $_key => $_value ) {
                if ( is_author( $_value ) ) {
                    $show = true; break;
                }
            }
        } else { $show = is_author( $value ); }

        return self::compare( $show, true, $operator );
    }

    /**
     * Render the widget output in the editor.
     *
     * Written as a Backbone JavaScript template and used to generate the live preview.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function content_template()
    {
        ?>
        <# var iconsHTML = {}; #>
        <div class="elementor-social-icons-wrapper elementor-grid">
            <# _.each( settings.floating_icon_list, function( item, index ) {
            var link = item.link ? item.link.url : '',
            migrated = elementor.helpers.isIconMigrated( item, 'floating_icon' );
            floating = elementor.helpers.getSocialNetworkNameFromIcon( item.floating_icon, item.floating, false, migrated );
            #>
            <div class="elementor-grid-item">
                <a class="elementor-icon elementor-social-icon elementor-social-icon-{{ floating }} elementor-animation-{{ settings.hover_animation }} elementor-repeater-item-{{item._id}}" href="{{ link }}">
                    <span class="elementor-screen-only">{{{ floating }}}</span>
                    <#
                    iconsHTML[ index ] = elementor.helpers.renderIcon( view, item.floating_icon, {}, 'i', 'object' );
                    if ( ( ! item.floating || migrated ) && iconsHTML[ index ] && iconsHTML[ index ].rendered ) { #>
                    {{{ iconsHTML[ index ].value }}}
                    <# } else { #>
                    <i class="{{ item.floating }}"></i>
                    <# }
                    #>
                </a>
            </div>
            <# } ); #>
        </div>
        <?php
    }
}
