<?php

namespace ElementorFloatingMenu;

use ElementorFloatingMenu\PageSettings\Page_Settings;

/**
 * Class Plugin
 *
 * Main Plugin class
 * @since 1.2.0
 */
class Plugin
{

    /**
     * Instance
     *
     * @since 1.2.0
     * @access private
     * @static
     *
     * @var Plugin The single instance of the class.
     */
    private static $_instance = null;

    /**
     * Instance
     *
     * Ensures only one instance of the class is loaded or can be loaded.
     *
     * @return Plugin An instance of the class.
     * @since 1.2.0
     * @access public
     *
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

    /**
     * widget_scripts
     *
     * Load required plugin core files.
     *
     * @since 1.2.0
     * @access public
     */
    public function widget_scripts()
    {
        wp_register_script(
            'elementor-floating-menu',
            plugins_url('/assets/js/floating-menu.js', __FILE__),
            ['jquery'],
            false,
            true
        );
    }

    /**
     * Editor scripts
     *
     * Enqueue plugin javascripts integrations for Elementor editor.
     *
     * @since 1.2.1
     * @access public
     */
    public function editor_scripts()
    {
        add_filter('script_loader_tag', [$this, 'editor_scripts_as_a_module'], 10, 2);

        wp_enqueue_script(
            'elementor-floating-menu-editor',
            plugins_url('/assets/js/editor/editor.js', __FILE__),
            [
                'elementor-editor',
            ],
            '1.2.1',
            true
        );
    }

    /**
     * Force load editor script as a module
     *
     * @param  string  $tag
     * @param  string  $handle
     *
     * @return string
     * @since 1.2.1
     *
     */
    public function editor_scripts_as_a_module($tag, $handle)
    {
        if ('elementor-floating-menu-editor' === $handle) {
            $tag = str_replace('<script', '<script type="module"', $tag);
        }

        return $tag;
    }

    /**
     * Include Widgets files
     *
     * Load widgets files
     *
     * @since 1.2.0
     * @access private
     */
    private function include_widgets_files()
    {
        require_once(__DIR__.'/widgets/floating-menu.php');
    }

    /**
     * Register Widgets
     *
     * Register new Elementor widgets.
     *
     * @since 1.2.0
     * @access public
     */
    public function register_widgets()
    {
        // Its is now safe to include Widgets files
        $this->include_widgets_files();

        // Register Widgets
        \Elementor\Plugin::instance()->widgets_manager->register_widget_type(new Widgets\Floating_Menu());
    }

    /**
     * Register CSS
     *
     * Register new css.
     *
     * @since 1.2.0
     * @access public
     */
    public function widget_styles()
    {
        wp_register_style('floating-menu-css', plugins_url('/assets/css/floating-menu.css', __FILE__));
        wp_enqueue_style('floating-menu-css');
    }

    /**
     * Add page settings controls
     *
     * Register new settings for a document page settings.
     *
     * @since 1.2.1
     * @access private
     */
    private function add_page_settings_controls()
    {
        require_once(__DIR__.'/page-settings/manager.php');
        new Page_Settings();
    }

    /**
     *  Plugin class constructor
     *
     * Register plugin action hooks and filters
     *
     * @since 1.2.0
     * @access public
     */
    public function __construct()
    {
        // Register widget scripts
        add_action('elementor/frontend/after_register_scripts', [$this, 'widget_scripts']);

        // Register widgets
        add_action('elementor/widgets/widgets_registered', [$this, 'register_widgets']);

        // Register editor scripts
        add_action('elementor/editor/after_enqueue_scripts', [$this, 'editor_scripts']);

        // Register css
        add_action('elementor/frontend/after_enqueue_styles', [$this, 'widget_styles']);

        $this->add_page_settings_controls();
    }
}

// Instantiate Plugin Class
Plugin::instance();
